<?php
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/helpers.php';
$page_title = 'New Invoice';
$db = db();

$customers = all($db, "SELECT id,name FROM customers ORDER BY name ASC");
$items = all($db, "SELECT id,name FROM items ORDER BY name ASC");

function next_invoice_no(mysqli $db): string {
  $row = one($db, "SELECT COALESCE(MAX(id),0)+1 AS n FROM invoices");
  $n = (int)($row['n'] ?? 1);
  return 'INV' . date('Ymd') . '-' . str_pad((string)$n, 4, '0', STR_PAD_LEFT);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_check();
  $customer_id = (int)($_POST['customer_id'] ?? 0);
  $invoice_date = $_POST['invoice_date'] ?? date('Y-m-d');
  $invoice_no = trim($_POST['invoice_no'] ?? '');
  if ($invoice_no === '') $invoice_no = next_invoice_no($db);

  $line_item_id = $_POST['item_id'] ?? [];
  $line_uom_id  = $_POST['uom_id'] ?? [];
  $line_qty     = $_POST['qty'] ?? [];
  $line_price   = $_POST['price'] ?? [];

  if ($customer_id <= 0) {
    flash_set('danger', 'Select a customer.');
  } else {
    $db->begin_transaction();
    try {
      $total = 0.0;

      $st = $db->prepare("INSERT INTO invoices(customer_id,shift_id,invoice_no,invoice_date,total,balance,status) VALUES(?,?,?,?,?,?,?)");
      $zero = 0.0; $status = 'Unpaid';
      $shift_id = (int)($_SESSION["shift_id"] ?? 0); if ($shift_id<=0) $shift_id = null;
      $st->bind_param("iissdds", $customer_id, $shift_id, $invoice_no, $invoice_date, $zero, $zero, $status);
      $st->execute();
      $invoice_id = $st->insert_id;
      $st->close();

      $ins = $db->prepare("INSERT INTO invoice_items(invoice_id,item_id,uom_id,qty,price,line_total,cost_total,profit) VALUES(?,?,?,?,?,?,?,?)");

      for ($i=0; $i<count($line_item_id); $i++) {
        $item_id = (int)$line_item_id[$i];
        $uom_id  = (int)($line_uom_id[$i] ?? 0);
        $qty     = (float)($line_qty[$i] ?? 0);
        $price   = (float)($line_price[$i] ?? 0);

        if ($item_id<=0 || $uom_id<=0 || $qty<=0) continue;

        // Trust posted price but you can also lock to DB:
        $u = one($db, "SELECT price FROM item_uoms WHERE id=? AND item_id=?","ii",[$uom_id,$item_id]);
        if ($u) $price = (float)$u['price'];


        $lt = $qty * $price;
        $total += $lt;

        // Stock + Profit (simple)
        $u2 = one($db, "SELECT conversion_qty, cost FROM item_uoms WHERE id=? AND item_id=?","ii",[$uom_id,$item_id]);
        $conv_qty = (float)($u2['conversion_qty'] ?? 1);
        if ($conv_qty <= 0) $conv_qty = 1;
        $qty_base = $qty * $conv_qty;

        $on = one($db, "SELECT avg_cost_base FROM stock_onhand WHERE item_id=?","i",[$item_id]);
        $avg_cost_base = (float)($on['avg_cost_base'] ?? 0);

        if ($avg_cost_base <= 0.0000001) {
          // fallback to default cost from UOM
          $fallback_cost_uom = (float)($u2['cost'] ?? 0);
          $avg_cost_base = $fallback_cost_uom / $conv_qty;
        }

        $cost_total = $qty_base * $avg_cost_base;
        $profit = $lt - $cost_total;

        $ins->bind_param("iiiddddd", $invoice_id, $item_id, $uom_id, $qty, $price, $lt, $cost_total, $profit);
        $ins->execute();

        // Update stock onhand and movement (no negative-stock prevention in this simple MVP)
        $db->query("INSERT INTO stock_onhand(item_id, qty_base, avg_cost_base) VALUES ($item_id, 0, 0)
                    ON DUPLICATE KEY UPDATE item_id=item_id");
        $db->query("UPDATE stock_onhand SET qty_base = qty_base - ".((float)$qty_base)." WHERE item_id = $item_id");
        $db->query("INSERT INTO stock_moves(move_date,move_type,ref_type,ref_id,item_id,qty_base,cost_base,note)
                    VALUES (NOW(),'SALE','invoice',$invoice_id,$item_id,".((float)(-$qty_base)).",".((float)$avg_cost_base).",'Sale')");

      }
      $ins->close();

      $balance = $total;
      $status = ($balance<=0.0001) ? 'Paid' : 'Unpaid';

      $up = $db->prepare("UPDATE invoices SET total=?, balance=?, status=? WHERE id=?");
      $up->bind_param("ddsi", $total, $balance, $status, $invoice_id);
      $up->execute(); $up->close();

      $db->commit();
      flash_set('success', 'Invoice created.');
      header('Location: ' . base_url() . '/public/invoice_view.php?id=' . $invoice_id);
      exit;
    } catch (Throwable $e) {
      $db->rollback();
      flash_set('danger', 'Failed: ' . $e->getMessage());
    }
  }
}

$inv_no = next_invoice_no($db);

include __DIR__ . '/../includes/header.php';
?>
<div class="row g-3">
  <div class="col-12">
    <div class="card p-3">
      <div class="d-flex justify-content-between align-items-center">
        <h4 class="mb-0">New invoice</h4>
        <a class="btn btn-outline-light" href="<?= base_url() ?>/public/invoices.php">Back</a>
      </div>

      <form method="post" class="mt-3" id="invForm">
        <?= csrf_field() ?>
        <div class="row g-2">
          <div class="col-12 col-md-5">
            <label class="form-label">Customer</label>
            <select class="form-select" name="customer_id" required>
              <option value="">Select customer</option>
              <?php foreach($customers as $c): ?>
                <option value="<?= (int)$c['id'] ?>"><?= h($c['name']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-6 col-md-3">
            <label class="form-label">Date</label>
            <input class="form-control" type="date" name="invoice_date" value="<?= h(date('Y-m-d')) ?>" required>
          </div>
          <div class="col-6 col-md-4">
            <label class="form-label">Invoice no</label>
            <input class="form-control" name="invoice_no" value="<?= h($inv_no) ?>">
          </div>
        </div>

        <hr class="border-secondary my-3">

        <div class="row g-2 mt-3">
          <div class="col-12 col-md-6">
            <label class="form-label">Barcode quick add (optional)</label>
            <input class="form-control" id="barcodeInput" placeholder="Scan barcode then Enter">
            <div class="small opacity-75">If barcode matches an item, a new line is added automatically.</div>
          </div>
          <div class="col-12 col-md-6 text-md-end">
            <div class="small opacity-75">Current shift</div>
            <div class="fw-semibold" id="shiftBadge"><?= isset($_SESSION['shift_id']) ? 'Shift #'.(int)$_SESSION['shift_id'].' (Open)' : 'No open shift' ?></div>
            <div class="small opacity-75">Shifts are optional in this MVP.</div>
          </div>
        </div>


        <div class="table-responsive">
          <table class="table table-sm align-middle" id="lines">
            <thead>
              <tr>
                <th style="min-width:220px;">Item</th>
                <th style="min-width:180px;">UOM</th>
                <th class="text-end" style="width:120px;">Qty</th>
                <th class="text-end" style="width:140px;">Price</th>
                <th class="text-end" style="width:160px;">Line Total</th>
                <th style="width:60px;"></th>
              </tr>
            </thead>
            <tbody></tbody>
          </table>
        </div>

        <div class="d-flex gap-2">
          <button type="button" class="btn btn-outline-info" onclick="addRow()">+ Add line</button>
          <button class="btn btn-primary ms-auto">Save invoice</button>
        </div>

        <div class="mt-3 text-end">
          <div class="opacity-75 small">Grand total</div>
          <div class="fs-3 fw-bold" id="grandTotal">KES 0.00</div>
        </div>
      </form>
    </div>
  </div>
</div>

<script>
const ITEMS = <?= json_encode($items, JSON_UNESCAPED_UNICODE) ?>;

function money(n){ return (Number(n)||0).toLocaleString(undefined,{minimumFractionDigits:2,maximumFractionDigits:2}); }

function addRow(){
  const tbody = document.querySelector('#lines tbody');
  const tr = document.createElement('tr');
  tr.innerHTML = `
    <td>
      <select class="form-select form-select-sm itemSel" name="item_id[]" required>
        <option value="">Select item</option>
        ${ITEMS.map(it=>`<option value="${it.id}">${escapeHtml(it.name)}</option>`).join('')}
      </select>
    </td>
    <td>
      <select class="form-select form-select-sm uomSel" name="uom_id[]" required disabled>
        <option value="">Select UOM</option>
      </select>
    </td>
    <td><input class="form-control form-control-sm text-end qty" name="qty[]" type="number" step="0.001" value="1" min="0.001" required></td>
    <td><input class="form-control form-control-sm text-end price" name="price[]" type="number" step="0.01" value="0" min="0"></td>
    <td class="text-end fw-semibold lineTotal">KES 0.00</td>
    <td class="text-end"><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calc();">×</button></td>
  `;
  tbody.appendChild(tr);

  const itemSel = tr.querySelector('.itemSel');
  const uomSel = tr.querySelector('.uomSel');
  itemSel.addEventListener('change', async () => {
    uomSel.innerHTML = '<option value="">Loading...</option>';
    uomSel.disabled = true;
    const itemId = itemSel.value;
    if(!itemId){ uomSel.innerHTML='<option value="">Select UOM</option>'; return; }
    const res = await fetch('<?= base_url() ?>/public/api/uoms.php?item_id=' + encodeURIComponent(itemId));
    const data = await res.json();
    const uoms = (data && data.uoms) ? data.uoms : [];
    uomSel.innerHTML = '<option value="">Select UOM</option>' + uoms.map(u=>`<option value="${u.id}" data-price="${u.price}">${escapeHtml(u.uom_name)}</option>`).join('');
    uomSel.disabled = false;
  });

  uomSel.addEventListener('change', () => {
    const opt = uomSel.options[uomSel.selectedIndex];
    const price = opt ? opt.getAttribute('data-price') : '0';
    tr.querySelector('.price').value = price || 0;
    calc();
  });

  tr.querySelector('.qty').addEventListener('input', calc);
  tr.querySelector('.price').addEventListener('input', calc);

  calc();
}

function calc(){
  let grand = 0;
  document.querySelectorAll('#lines tbody tr').forEach(tr=>{
    const qty = Number(tr.querySelector('.qty').value)||0;
    const price = Number(tr.querySelector('.price').value)||0;
    const lt = qty * price;
    grand += lt;
    tr.querySelector('.lineTotal').textContent = 'KES ' + money(lt);
  });
  document.getElementById('grandTotal').textContent = 'KES ' + money(grand);
}

function escapeHtml(s){
  return String(s).replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#039;'}[m]));
}


const barcodeInput = document.getElementById('barcodeInput');
if (barcodeInput){
  barcodeInput.addEventListener('keydown', async (e)=>{
    if (e.key === 'Enter'){
      e.preventDefault();
      const code = barcodeInput.value.trim();
      if(!code) return;
      const res = await fetch('<?= base_url() ?>/public/api/item_by_barcode.php?barcode=' + encodeURIComponent(code));
      const data = await res.json();
      if (data && data.ok){
        // add a row, pick item, fill uoms, select first uom
        addRow();
        const tr = document.querySelector('#lines tbody tr:last-child');
        const itemSel = tr.querySelector('.itemSel');
        itemSel.value = data.item.id;
        itemSel.dispatchEvent(new Event('change'));
        // wait a bit for uoms to load then select first
        setTimeout(()=>{
          const uomSel = tr.querySelector('.uomSel');
          if (uomSel && data.uoms && data.uoms.length){
            uomSel.value = data.uoms[0].id;
            uomSel.dispatchEvent(new Event('change'));
          }
          tr.querySelector('.qty').focus();
          barcodeInput.value = '';
        }, 250);
      } else {
        alert('Barcode not found: ' + code);
      }
    }
  });
}

addRow();
</script>
<?php include __DIR__ . '/../includes/footer.php'; ?>
