<?php
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/helpers.php';
$page_title = 'Receive GRN';
$db = db();

$items = all($db, "SELECT id,name FROM items ORDER BY name ASC");

function next_grn_no(mysqli $db): string {
  $row = one($db, "SELECT COALESCE(MAX(id),0)+1 AS n FROM grns");
  $n = (int)($row['n'] ?? 1);
  return 'GRN' . date('Ymd') . '-' . str_pad((string)$n, 4, '0', STR_PAD_LEFT);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_check();
  $grn_no = trim($_POST['grn_no'] ?? '');
  $grn_date = $_POST['grn_date'] ?? date('Y-m-d');
  $supplier = trim($_POST['supplier'] ?? '');
  $notes = trim($_POST['notes'] ?? '');

  $line_item_id = $_POST['item_id'] ?? [];
  $line_uom_id  = $_POST['uom_id'] ?? [];
  $line_qty     = $_POST['qty'] ?? [];
  $line_cost    = $_POST['unit_cost'] ?? [];

  if ($grn_no === '') $grn_no = next_grn_no($db);

  $db->begin_transaction();
  try {
    $uid = (int)($_SESSION['user_id'] ?? 0);
    $st = $db->prepare("INSERT INTO grns(grn_no,grn_date,supplier,notes,created_by) VALUES(?,?,?,?,?)");
    $st->bind_param("ssssi", $grn_no, $grn_date, $supplier, $notes, $uid);
    $st->execute();
    $grn_id = $st->insert_id;
    $st->close();

    $ins = $db->prepare("INSERT INTO grn_items(grn_id,item_id,uom_id,qty,unit_cost,line_total) VALUES(?,?,?,?,?,?)");

    for ($i=0; $i<count($line_item_id); $i++) {
      $item_id = (int)$line_item_id[$i];
      $uom_id  = (int)($line_uom_id[$i] ?? 0);
      $qty     = (float)($line_qty[$i] ?? 0);
      $unit_cost = (float)($line_cost[$i] ?? 0);
      if ($item_id<=0 || $uom_id<=0 || $qty<=0) continue;

      $u = one($db, "SELECT conversion_qty FROM item_uoms WHERE id=? AND item_id=?","ii",[$uom_id,$item_id]);
      $conv = (float)($u['conversion_qty'] ?? 1);
      if ($conv<=0) $conv=1;

      $line_total = $qty * $unit_cost;
      $ins->bind_param("iiiddd", $grn_id, $item_id, $uom_id, $qty, $unit_cost, $line_total);
      $ins->execute();

      // Update stock avg cost (weighted average at base unit)
      $qty_base = $qty * $conv;
      $cost_base_in = ($unit_cost / $conv);

      $db->query("INSERT INTO stock_onhand(item_id, qty_base, avg_cost_base) VALUES ($item_id, 0, 0)
                  ON DUPLICATE KEY UPDATE item_id=item_id");
      $on = one($db, "SELECT qty_base, avg_cost_base FROM stock_onhand WHERE item_id=?","i",[$item_id]);
      $old_qty = (float)($on['qty_base'] ?? 0);
      $old_avg = (float)($on['avg_cost_base'] ?? 0);

      $new_qty = $old_qty + $qty_base;
      if ($new_qty > 0.0001) {
        $new_avg = (($old_qty * $old_avg) + ($qty_base * $cost_base_in)) / $new_qty;
      } else {
        $new_avg = $cost_base_in;
      }

      $db->query("UPDATE stock_onhand SET qty_base = qty_base + ".((float)$qty_base).", avg_cost_base = ".((float)$new_avg)." WHERE item_id = $item_id");
      $db->query("INSERT INTO stock_moves(move_date,move_type,ref_type,ref_id,item_id,qty_base,cost_base,note)
                  VALUES (NOW(),'GRN','grn',$grn_id,$item_id,".((float)$qty_base).",".((float)$cost_base_in).",'GRN Receive')");
    }
    $ins->close();

    $db->commit();
    flash_set('success','GRN received and stock updated.');
    header('Location: ' . base_url() . '/public/stock_onhand.php');
    exit;
  } catch (Throwable $e) {
    $db->rollback();
    flash_set('danger','Failed: '.$e->getMessage());
  }
}

$grn_no = next_grn_no($db);

include __DIR__ . '/../includes/header.php';
?>
<div class="card p-3">
  <div class="d-flex justify-content-between align-items-center">
    <h4 class="mb-0">Receive GRN</h4>
    <a class="btn btn-outline-light" href="<?= base_url() ?>/public/stock_onhand.php">Back</a>
  </div>

  <form method="post" class="mt-3" id="grnForm">
    <?= csrf_field() ?>
    <div class="row g-2">
      <div class="col-12 col-md-4">
        <label class="form-label">GRN No</label>
        <input class="form-control" name="grn_no" value="<?= h($grn_no) ?>">
      </div>
      <div class="col-6 col-md-3">
        <label class="form-label">Date</label>
        <input class="form-control" type="date" name="grn_date" value="<?= h(date('Y-m-d')) ?>" required>
      </div>
      <div class="col-6 col-md-5">
        <label class="form-label">Supplier (optional)</label>
        <input class="form-control" name="supplier" placeholder="e.g. ABC Suppliers Ltd">
      </div>
    </div>

    <div class="mb-2 mt-2">
      <label class="form-label">Notes</label>
      <input class="form-control" name="notes" placeholder="optional">
    </div>

    <hr class="border-secondary my-3">

    <div class="table-responsive">
      <table class="table table-sm align-middle" id="lines">
        <thead>
          <tr>
            <th style="min-width:220px;">Item</th>
            <th style="min-width:180px;">UOM</th>
            <th class="text-end" style="width:120px;">Qty</th>
            <th class="text-end" style="width:140px;">Unit Cost</th>
            <th class="text-end" style="width:160px;">Line Total</th>
            <th style="width:60px;"></th>
          </tr>
        </thead>
        <tbody></tbody>
      </table>
    </div>

    <div class="d-flex gap-2">
      <button type="button" class="btn btn-outline-info" onclick="addRow()">+ Add line</button>
      <button class="btn btn-primary ms-auto">Save GRN</button>
    </div>

    <div class="mt-3 text-end">
      <div class="opacity-75 small">Grand total (cost)</div>
      <div class="fs-3 fw-bold" id="grandTotal">KES 0.00</div>
    </div>
  </form>
</div>

<script>
const ITEMS = <?= json_encode($items, JSON_UNESCAPED_UNICODE) ?>;
function money(n){ return (Number(n)||0).toLocaleString(undefined,{minimumFractionDigits:2,maximumFractionDigits:2}); }
function escapeHtml(s){ return String(s).replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#039;'}[m])); }

async function fetchUoms(itemId){
  const res = await fetch('<?= base_url() ?>/public/api/uoms.php?item_id=' + encodeURIComponent(itemId));
  const data = await res.json();
  return (data && data.uoms) ? data.uoms : [];
}

function addRow(){
  const tbody = document.querySelector('#lines tbody');
  const tr = document.createElement('tr');
  tr.innerHTML = `
    <td>
      <select class="form-select form-select-sm itemSel" name="item_id[]" required>
        <option value="">Select item</option>
        ${ITEMS.map(it=>`<option value="${it.id}">${escapeHtml(it.name)}</option>`).join('')}
      </select>
    </td>
    <td>
      <select class="form-select form-select-sm uomSel" name="uom_id[]" required disabled>
        <option value="">Select UOM</option>
      </select>
    </td>
    <td><input class="form-control form-control-sm text-end qty" name="qty[]" type="number" step="0.001" value="1" min="0.001" required></td>
    <td><input class="form-control form-control-sm text-end cost" name="unit_cost[]" type="number" step="0.01" value="0" min="0"></td>
    <td class="text-end fw-semibold lineTotal">KES 0.00</td>
    <td class="text-end"><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calc();">×</button></td>
  `;
  tbody.appendChild(tr);

  const itemSel = tr.querySelector('.itemSel');
  const uomSel = tr.querySelector('.uomSel');

  itemSel.addEventListener('change', async () => {
    uomSel.innerHTML = '<option value="">Loading...</option>';
    uomSel.disabled = true;
    const itemId = itemSel.value;
    if(!itemId){ uomSel.innerHTML='<option value="">Select UOM</option>'; return; }
    const uoms = await fetchUoms(itemId);
    uomSel.innerHTML = '<option value="">Select UOM</option>' + uoms.map(u=>`<option value="${u.id}">${escapeHtml(u.uom_name)}</option>`).join('');
    uomSel.disabled = false;
  });

  tr.querySelector('.qty').addEventListener('input', calc);
  tr.querySelector('.cost').addEventListener('input', calc);
  calc();
}

function calc(){
  let grand = 0;
  document.querySelectorAll('#lines tbody tr').forEach(tr=>{
    const qty = Number(tr.querySelector('.qty').value)||0;
    const cost = Number(tr.querySelector('.cost').value)||0;
    const lt = qty * cost;
    grand += lt;
    tr.querySelector('.lineTotal').textContent = 'KES ' + money(lt);
  });
  document.getElementById('grandTotal').textContent = 'KES ' + money(grand);
}

addRow();
</script>
<?php include __DIR__ . '/../includes/footer.php'; ?>
